#!/bin/bash --norc
# *******************************************************************************
# * MODULE     : mkqt.sh
# * DESCRIPTION: build the TeXmacs SDK Qt
# * COPYRIGHT  : (C) 2016-2019 Denis RAUX
# *******************************************************************************
# * This software falls under the GNU general public license version 3 or later.
# * It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
# * in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
# *******************************************************************************

. ./common.sh

[[ "$TMMODE" == a ]] && isDone qt && exit 0

trap "setStatus false qt;echo Qt build failed;exit 250" SIGINT

unset QMAKESPEC
typeset d f QTstr
typeset -i res QTver
shopt -s nullglob

QTstr="$(echo qt-everywhere-*src-*)"
QTstr=${QTstr##*-}
QTver=${QTstr%%.*}

unset {C,CPP,CXX,LD}FLAGS
#to build qmake
### for mac mimi test $QTver -lt 5 && checkflag -std=c99 && export OPENSOURCE_CXXFLAGS="-std=c99"

typeset qtopt="-opensource -static"

# qt  does not use the right parameter for a static version
export PKG_CONFIG="$(type -P pkg-config) --static"

# Windows Qt doesn't accept --prefix so we have to move to final destination
# paths arrangement according to the OS
case $TMOS in
win) 
  typeset dst
  test -n "$TMSDKDBG" && dst=/Qtg || dst=/Qt
  if [[ ${QTver:=0} == 0 ]]
  then test -f $dst/QTver.txt && QTver=$(<$dst/QTver.txt)
  else 
    echo $QTver >$dst/QTver.txt
    cat<<-EOF >$dst/qt.conf
      [Paths]
      Prefix=/$dst
		EOF
  fi
  test $(CompilVersion g++) -ge 800 && qtopt+=' -platform win32-g++-8.0'
  qtopt+=" -I $(cygpath -m $TMSDKPH/include) -L $(cygpath -m $TMSDKPH/lib)"
  ;;
mac)
	if test $QTver -lt 5
	then
		#have to set the right platform

		savCASE=$(shopt -p nocasematch)
		shopt -s nocasematch

		{ [[ "$CC" == clang || "$CC" == llvm || $(gcc --version 2>/dev/null) =~ clang || 
			$(gcc --version 2>/dev/null) =~ llvm ]]  && qtopt+=' -platform macx-llvm'; } ||
			{ test $(CompilVersion gcc) -ge 402 && qtopt+=' -platform macx-g++42'; } ||
			{ test $(CompilVersion gcc) -ge 400 && qtopt+=' -platform macx-g++40'; } ||
			qtopt+=' -platform macx-g++'
		$savCASE
	fi
  		test $QTver -eq 5 -a $(uname -m) == arm64 && qtopt+=" -platform unsupported/macx-clang-arm64"


		qtopt+=" -I $TMSDKPH/include -L $TMSDKPH/lib"
		# Qt offers no means to set search_paths_first
		# we nedd to force it. It alway looks for libz.dylib too
		export LDFLAGS+="-Wl,-search_paths_first $TMSDKPH/lib/libz.a"
;;
lin) # extra options for some libs for linux
  qtopt+=" $(pkg-config --cflags freetype2)"
  qtopt+=" -I $TMSDKPH/include -L $TMSDKPH/lib"
  qtopt+=" -fontconfig"
  ;;
esac

if [[ $QTver == 0 ]] ; then echo QT invalid version : $QTver ; exit 2;fi

# Some options need two dashes (qt config bug)
case $QTver in
4)
  qtopt+=" -system-libpng -qt-libjpeg -qt-libtiff -no-libmng"
  qtopt+=" -fast -exceptions -no-accessibility -no-qt3support"
  qtopt+=" -no-openssl -no-declarative-debug -no-scripttools -no-script"
  qtopt+=" -no-webkit -no-phonon -no-phonon-backend -no-multimedia -no-xmlpatterns"
  qtopt+=" -nomake imports -nomake demos -nomake examples -nomake doc -nomake tools"
  qtopt+=" -nomake translations -no-opengl"
	;;
5)
  qtopt+=" -pkg-config"
  qtopt+=" -nomake examples -nomake tests"
  qtopt+=" -no-glib --pcre=qt"
  qtopt+=" --no-opengl -no-xcb-xlib -no-gtk"
#  qtopt+=" -no-feature-accessibility" it's adviced to set it
  qtopt+=" -skip qt3d -skip qtactiveqt -skip qtandroidextras -skip qtcanvas3d -skip qtcharts"
  qtopt+=" -skip qtconnectivity -skip qtdatavis3d -skip qtdeclarative -skip qtdoc"
  qtopt+=" -skip qtgamepad -skip qtgraphicaleffects -skip qtmultimedia -skip qtpurchasing"
  qtopt+=" -skip qtwebchannel -skip qtwebengine -skip qtwebsockets -skip qtwebview"
  qtopt+=" -skip qtxmlpatterns"
  qtopt+=" -skip qtquickcontrols -skip qtquickcontrols2 -skip qt3d"
  qtopt+=" -skip qtxmlpatterns -skip qtserialbus -skip qtserialport -skip qtscript -no-sqlite -no-sql-mysql"
	;;
6)
  qtopt+=" -pkg-config -no-pch"
  qtopt+=" -nomake examples -nomake tests"
  qtopt+=" -no-glib --pcre=qt"
  qtopt+=" --no-opengl -no-xcb-xlib -no-gtk"
#  qtopt+=" -no-feature-accessibility" it's adviced to set it
  qtopt+=" -skip qt3d -skip qtactiveqt"
  qtopt+=" -skip qtcharts -skip qtcoap -skip qtconnectivity -skip qtdatavis3d"
  qtopt+=" -skip qtdeclarative -skip qt5compat"
  qtopt+=" -skip qtdoc"
  qtopt+=" -skip qtlottie -skip qtmqtt -skip qtmultimedia -skip qtnetworkauth -skip qtopcua"
  qtopt+=" -skip qtpositioning -skip qtquick3d -skip qtquicktimeline -skip qtremoteobjects"
  qtopt+=" -skip qtscxml -skip qtsensors -skip qtserialbus -skip qtserialport -skip qtshadertools"
  qtopt+=" -skip qttools -skip qttranslations -skip qtvirtualkeyboard"
  qtopt+=" -skip qtwebchannel -skip qtwebengine -skip qtwebsockets -skip qtwebview"
	;;
*)
	echo Qt version not supported
	exit 22
	;;
esac

case $TMOS in
mac) 
  qtopt+=" -prefix $TMSDKPH"
	d=${TMSDKDBG:+-debug -no-framework}
	d=${d:--release}
	qtopt+=" -no-sm -no-sse4.2 -no-sse4.1 -no-sse3"
	test -n "$xcodeSDK" && { test $QTver -eq 4 && qtopt+=" -sdk $xcodeSDK" || qtopt+=" -sdk macosx$vsdk"; }
	export QMAKE_MACOSX_DEPLOYMENT_TARGET=$MACOSX_DEPLOYMENT_TARGET
  if [[  $QTver == 4  ]]
  then 
    qtopt+=" -no-sm -no-javascript-jit -no-xinerama -make libs"
  fi
	cd  qt-* || exit 20
	echo "./configure  $qtopt $d" >config.log
	if test $QTver -ne 6
	then echo yes | ./configure  $qtopt $d  2>&1 |tee -a config.log && make -j$cntproc && make install
	else echo "if you want to redo configure, delete qt*-src-*/CMakeCache.txt" 
		(test -f CMakeCache.txt || ./configure  $qtopt $d  2>&1 |tee -a config.log) &&
		cmake --build . --parallel && cmake --install . --prefix $(tmsdk)
	fi
	res=$?
	;;
lin) # we need to add libpng which isn't in dependencies list 
	typeset edit=$(type -P sed) 2>/dev/null
  qtopt+=" -prefix $TMSDKPH"
	d=${TMSDKDBG:+-debug}
	d=${d:--release}
	qtopt+=" -no-sm -no-sse4.2 -no-sse4.1 -no-sse3"
  if [[  $QTver == 4  ]]
  then 
    qtopt+=" -no-sm -no-javascript-jit -no-xinerama -make libs"
  fi
	cd  qt-* || exit 20
	echo "./configure  $qtopt $d" >config.log
	echo yes | ./configure  $qtopt $d  2>&1 |tee -a config.log && \
		make -j$cntproc && \
		make install && { test $QTver -eq 4 && ${edit:=ed} -f/dev/fd/3 3<<-EOF $TMSDKPH/lib/libQtGui.prl || : ; }
			s/-lfreetype/-lfreetype -lpng/g
			wq
			EOF
	 res=$?
	;;
win)

	#qtopt+=" -platform win32-g++-4.6 -host-little-endian"
	d=${TMSDKDBG:+-debug}
	d=${d:--release}
	cd $dst || exit 20
	# there are some bugs in the path given by qmake its location is not set
	cat >qt.conf <<-EOF
		[Paths]
		Prefix=$dst
	EOF
	echo "./configure  $qtopt $d" >config.log
	echo "echo y|configure $qtopt $d"|cmd  2>&1 |tee -a config.log && mingw32-make -j$cntproc
	res=$?

	#we need to correct the badly built pkconfig files
	if test $res -eq 0
	then 
		typeset basedir=$(pwd -W)
		basedir=${basedir%$PWD}
		pushd $TMSDKPH/lib || { echo LIBS not built; exit 10; }
		for f in *.prl
		do pc=$(<$f)
			echo "${pc//${basedir}/}" > $f
		done
		pushd pkgconfig || { echo pkgdir not built; exit 20; }
		for f in *.pc
		do pc=$(<$f)
			pc=${pc/-l*pkgconfig\\/-l}
			pc=${pc//\\/\/}
			echo "${pc//${basedir}/}" > $f
		done
		popd
		popd
	fi	
	;;
*) echo OS non supported; exit 1;;
esac

if test $res -eq 0
then
	# rename the debug libraries
	if test -n "$TMSDKDBG"
	then	
	  typeset f
		pushd ${dst:=$TMSDKPH}/lib || { echo LIBS not built; exit 11; }
		shopt -s nullglob
		for f in libQt*_debug.a
		do  test -e ${f%_debug.a}.a || ln $f ${f%_debug.a}.a
		done
		for f in libQt*d.a libqt*d.a
    do  test -e ${f%d.a}.a || ln $f ${f%d.a}.a
		done
		popd
	fi
	setStatus true qt
else
	setStatus false qt
  echo Qt build failed
  exit 21
fi
exit 0
