#!/bin/bash
#*******************************************************************************
#* MODULE     : setenv.sh
#* DESCRIPTION: set the TeXmacs SDK environment
#* COPYRIGHT  : (C) 2016-2019 Denis RAUX
#*******************************************************************************
#* This software falls under the GNU general public license version 3 or later.
#* It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
#* in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
#*******************************************************************************
unset  s d TMOS ci opt mode l  prof

pushd $(dirname $0) >/dev/null
export SDKsrc=$(pwd)

declare fgdb
declare -i cntproc
# export for early calls
export TMOS cntproc fdone osxmin vsdk

unset   #in case set usefull for manual mode

########### General functions #############
function checkflag {
  gcc -E $1 - >/dev/null <<< "int main() {return 0; }"
	return $?
}

function clearflags {
  typeset f FLAGLIST="CFLAGS CXXFLAGS LDFLAGS LIBS"
  
  for f in $FLAGLIST
  do  if [[ -n "${!f}" ]]
      then typeset answer
      read -t 3 -n 1 -p "$f is ${!f} would you like to clear it ? (y/n)" answer
        if [[ $answer = y ]] 
        then unset $f
          echo " $f Cleared"
        fi
        echo
      fi
  done
}

function isDone {
  local -i ret=1
  while read module
  do  [[ "#+$1" == $module ]] && ret=0
      [[ "#-$1" == $module ]] && ret=1
  done 2>/dev/null <$fdone
  return $ret
}

function setStatus {
# $1 true or false 
# $2 module
  $1 && echo "#+$2" >>$fdone || echo "#-$2" >>$fdone
}

function setVar {
# $1 var name
# $2 value
  [[ $(<setenv.sh) =~ $1= ]] && echo "$1=$2" >>$fdone ||
    echo Variable already defined >&2
}

function CompilVersion {
  local out CCversion
  if out=$(mktemp -t tmsdk.XXXX)
  then $1 -o $out -xc - <<-"EOF"
    #include <stdlib.h>
    #include <stdio.h>
    int main () 
    {
      printf("%d",__GNUC__ * 100 + __GNUC_MINOR__);
      return 0;
    }
		EOF
    test $? -eq 0 && CCversion=$($out)
    rm $out
  fi
  echo $CCversion
}

# substitute $2 with $3 in file $1
function SubstInFile {
  local savIFS=$IFS; IFS=$'\n'
  typeset -a f
  f=($(<$1)) && echo "${f[*]/$2/$3}" >$1
  IFS=$savIFS
}

function OsType {
  local ci=$(shopt -p nocasematch)
  if test -n "$TMOS"
	then echo $TMOS
	else
		shopt -s nocasematch
		case $(uname -s) in
			Darwin) echo mac;;
			MINGW*) echo win;;
			Linux)  echo lin;;
			*) echo environment no supported
				exit;;
		esac
		$ci
	 fi
}

function CPifNewer {
  test ! -x $2/$(basename $1) -o $1 -nt $2/$(basename $1) && cp $1 $2/
  return $?
}
#####################################
function help {
  echo "Set the SDK building environment according to your operating system\n
Setup your env :
-c : check your environment and install package for Mingw
-4 : get the TAR archive for qt4 
-5 : get the TAR archive for qt5 
-6 : get the TAR archive for qt6
-g : update de SDKsrc scripts
		
Build the SDK		
-a automatic build
-m manual build
-S libraries status
-L list the different build status
-D display compiler define
-v display sdk and packages version

options :
Some option are only relevant in specific OSes

-l MacOSX : targeted level (10.6,10.7,10.8,..,11.1,..) needs -s
-s sdk    : sdk to use in /Developer/SDKs (10.6,10.7,10.8,...)
-p nb     : limits parallel build to nb processors 
-d        : debug build
"
test $(OsType) == "mac" && macSDKfullpath list
	
  exit 1
}

function macSDKfullpath {
		if test "$1" == "list"
	then
		echo -n "SDK locations : "
		ls   $(dirname $(xcrun --show-sdk-path))
		echo -n "current xcode SDK :  "
		xcrun --show-sdk-version
	else
	
		echo $(dirname $(xcrun --show-sdk-path))/MacOSX$1.sdk
	fi	
}

function CheckPrereq {
  local missing prof=etc/profile.d/texmacs.sh
  
  test -d bin || mkdir bin
  case $(OsType) in
  win)
    test -d ../TAR || echo "../TAR diretory not found use (-4,-5 or -6) option to create it"
    local pkglst="wget rsync subversion tar diffutils patch make autoconf automake-wrapper"
    pkglst+=" $(echo $MINGW_PACKAGE_PREFIX-{libtool-dev,make,pkg-config,gcc,gettext})"
    pacman --needed -S $pkglst

    echo -n "Creating profile..."
    if test -w $prof
    then cat<<"EOF" >$prof
export TM_MK_XPERT=c                                                                                                                                                                                         
export GUILE_LOAD_PATH=/SDK/share/guile/1.8/                                                                                                                                                                 
export with_tmrepo=/SDK                                                                                                                                                                                           
PATH=/Qt/bin:/InnoSetup6:$PATH 
export QMAKESPEC=/Qt/mkspecs/default
PKG_CONFIG_PATH="${MINGW_PREFIX}/share/aclocal:$PKG_CONFIG_PATH"
PATH=/Qt/bin:/InnoSetup6:$PATH
if ! test -f /Qt/qt.conf
then
  cat > /Qt/qt.conf <-EOF 
    [Paths]
    Prefix=$(dirname $(dirname $(pwd -W)))/Qt
  EOF
fi

EOF
      echo "done"
    else echo "file is readonly"
    fi
   mkdir -p /usr/local/bin
    CPifNewer ../TAR/tools/ntldd.exe /usr/local/bin/
    return 2
    ;;
  mac) return 1;;
  linux) return 1;;
  *) echo Not implemented yet;;
  esac
  return 0
}

function GetTar {
	test -d ../TAR && rm -r ../TAR/*
	curl ftp://ftp.texmacs.org/TeXmacs/dev/SDK/tmTAR$1.tar |tar -x --directory=..
}

function GetSdk {
 	( while ps -p$$>/dev/null;do :;done;
		curl ftp://ftp.texmacs.org/TeXmacs/dev/SDK/tmSDK.tar |tar -x  --strip-components=1 &&
		test ${PIPESTATUS[0]} -eq 0 && echo done press return ||
		echo cannot get SDK, press return )&
}

function DisplayVersions {
	local remSDKver=$(curl -s ftp://ftp.texmacs.org/TeXmacs/dev/SDK/SDKver)
	local remTARver=$(curl -s ftp://ftp.texmacs.org/TeXmacs/dev/SDK/TARver)
	local SDKver=$(cat SDKver 2>/dev/null)
	local TARver=$(cat ../TAR/TARver 2>/dev/null)
	local unk="???"
echo "
          installed remote
SDK       ${SDKver:-$unk}       ${remSDKver:-$unk}
PACKAGES  ${TARver:-$unk}       ${remTARver:-$unk}

CHANGELOGS:
$(curl -s ftp://ftp.texmacs.org/TeXmacs/dev/SDK/CHANGELOGlibs)

$(curl -s ftp://ftp.texmacs.org/TeXmacs/dev/SDK/CHANGELOGsdk)
"
}

function setEnv {
  if test -z "$cntproc"
  then case $TMOS in
    (win) cntproc=$(nproc);;
    (lin|mac) cntproc=$(getconf _NPROCESSORS_ONLN);;
    (*) cntproc=1;;
    esac
  fi

  # memorize current pkgconfig location
  oldpkgconfig=${PKG_CONFIG_PATH:+:$PKG_CONFIG_PATH}
  if test -x "$(type -P pkg-config)"
  then  oldpkgdefault=$(pkg-config --variable pc_path pkg-config)
        if test -n "$oldpkgdefault"
        then oldpkgconfig+=":$oldpkgdefault"
        else # we have to add the defaults manuallly
          savIFS=$IFS;savGLOB=$(shopt -p nullglob)
          IFS=:; shopt -s nullglob
          set /usr/lib*/pkgconfig /usr/lib/*/pkgconfig /usr/share/{lib*,.}/pkgconfig
          oldpkgconfig+=":$*"
          IFS=$savIFS; $savGLOB
        fi
  fi

  # don't stack the SDK
  if test -n "$TMSDKPH"
  then 
    echo SDK already set to $TMSDKPH type exit to leave
    exit 1
  fi

  clearflags

  #path for local tools
  test -d bin && PATH=$PWD/bin:$PATH

  CMD='SDKsrc=$PWD TMMODE=$mode'
  test -n "$d" && { sdk+=g; QT+=g; CMD+=' TMSDKDBG=$d'; }

  case "$TMOS" in
  (mac)
    CF=$optim
    if [ -n "$vsdk" ]
    then
			# we need indicate where  build the texmacs sdk (sdk) location
			# and also where the Apple SDK (macsysroot) is located
			export macsysroot=$(macSDKfullpath $vsdk)
			export xcodeSDK=$macsysroot
			if [ -n "macsysroot" ]
			then
				export macsysroot

				sdk+=/MacOSX$vsdk
				CF+=" --sysroot=$macsysroot"
				LDFLAGS+=" -Wl,-syslibroot,$macsysroot"
				test -d $sdk/bin || mkdir $sdk/bin
				test -f $sdk/bin/pkg-config || ln -s $sdk/../bin/pkg-config $sdk/bin/pkg-config
			else echo macsysroot not set >&2
			fi
    fi
    if [ -z "$MACOSX_DEPLOYMENT_TARGET" ]
    then
      tmp=$(sw_vers -productVersion);
      tmp1=${tmp%%.*};tmp=${tmp#$tmp1.};tmp="$tmp1.${tmp%%.*}"
      MACOSX_DEPLOYMENT_TARGET=${osxmin:-$tmp}
      setVar MACOSX_DEPLOYMENT_TARGET $MACOSX_DEPLOYMENT_TARGET
    else
      if test "$MACOSX_DEPLOYMENT_TARGET" != "$osxmin"
      then 
        echo "MACOSX_DEPLOYMENT_TARGET already defined to$MACOSX_DEPLOYMENT_TARGET\
          delete the file macos?_done.txt and\
          remake the SDK if you want use another value"
      else
        CF+=" -mmacosx-version-min=$osxmin"
        LDFLAGS+=" -Wl,-macosx_version_min,$osxmin"
      fi
    fi
    export MACOSX_DEPLOYMENT_TARGET
    CMD+=' TMOSXVER=$l TMSDKPH=$sdk CPPFLAGS="$CPPFLAGS${CPPFLAGS:+ }-I$sdk/include"'
    CMD+=' PKG_CONFIG_PATH="$sdk/lib/pkgconfig$oldpkgconfig"'
    CMD+=' CXXFLAGS=$CXXFLAGS${CXXFLAGS:+ }"$CF" CFLAGS=$CFLAGS${CFLAGS:+ }"$CF"'
    CMD+=' LDFLAGS=$LDFLAGS${LDFLAGS:+ }"-Wl,-search_paths_first -L$sdk/lib" PATH="$sdk/bin:$PATH"'
  ;;
  (win)
    CF=$optim
    [[ $(uname -s) =~ NT-10 ]] && export GUILE_LOAD_PATH=$sdk/share/guile/1.8 # else guile stay stucked with windows10
    CMD+=' TMSDKPH=$sdk PKG_CONFIG_PATH="$sdk/lib/pkgconfig:$QT/lib/pkgconfig/$oldpkgconfig"'
    CMD+=' CPPFLAGS=$CPPFLAGS${CPPFLAGS:+ }"-I$sdk/include" CXXFLAGS=$CXXFLAGS${CXXFLAGS:+ }"$CF"'
    CMD+=' CFLAGS=$CFLAGS${CFLAGS:+ }"$CF" LDFLAGS=$LDFLAGS${LDFLAGS:+ }"-L$sdk/lib" PATH="$sdk/bin:$QT/bin:$PATH"'
    ;;
  (lin)
    test $(CompilVersion ${CC:-gcc}) -ge 400 && CF="$optim ${d:--O3 -fexpensive-optimizations} " ||
      CF=$optim

    CMD+=' TMSDKPH=$sdk PKG_CONFIG_PATH="$sdk/lib/pkgconfig$oldpkgconfig"'
    CMD+=' CPPFLAGS="$CPPFLAGS${CPPFLAGS:+ }-I$sdk/include" CXXFLAGS="$CXXFLAGS${CXXFLAGS:+ }$CF"'
    CMD+=' CFLAGS="$CFLAGS${CFLAGS:+ }$CF" LDFLAGS="$LDFLAGS${LDFLAGS:+ }-L$sdk/lib" PATH="$sdk/bin:$PATH"'
    ;;
  (*) echo OS $TMOS not supported;
        exit 2;;
  esac
}

function absPath {
  if [[ $1 =~ ^/ ]]
  then echo $1
  else
    pushd $1 >/dev/null
    echo $(pwd)
    popd >/dev/null
  fi
}

function tmsdk {
  echo $TMSDKPH
}

function status {
  pushd $SDKsrc >/dev/null
  local m
  local -i st=0
  
  echo Environment
  grep = $fdone
  echo Libraries status : 
  for m in $(./mkbasic.sh -l) staticlib ghostscript guile qt 
  do isDone $m && echo $m : OK || { echo $m : KO && st=1; }
  done
  popd >/dev/null
  return $st
}

function ListLibs {
  local list l
  while read line
  do
    [[ $line =~ \#[+-]([[:alpha:]]+) ]] && eval lib__${BASH_REMATCH[1]}=1
  done < $1
  for l in ${!lib__*}
  do list+="${l#lib__} "
      unset $l
  done
  echo $list
}

function checkVerNum {
  [[ $1 =~ ^[[:digit:]]+\.[[:digit:]]+$ ]]
}

declare fdone optim

test $(uname -m) != arm64 && optim="-msse2 "

while getopts "Lahl:s:dp:mSc456gvD" opt
do
	typeset -i ret=0
  case $opt in
  l) osxmin=$OPTARG; checkVerNum $OPTARG || ( echo "wrong value for -l :"$'\n';help );;
  s) vsdk=$OPTARG; checkVerNum $OPTARG || ( echo "wrong value for -s :"$'\n';help );;
  d) d+="-O0 -ggdb"; fgdb=gdb_;;
  p) cntproc=$OPTARG;;
  a|m|S) mode=$opt;;
  L) test "$(echo *done.txt)" != '*done.txt' && echo *done.txt
      exit 0;;
  c) mode=c;;
	4|5|6) if ! GetTar $opt;then ret=$?;  echo something got wrong, check ;fi
			 exit $ret;;
	g) if ! GetSdk;then ret=$?;  echo something got wrong, check ;fi
		 exit $ret;;
	v) DisplayVersions; exit 0;;
	D) gcc -dM -E - < /dev/null; exit 0;;
  h) help;;
  esac
done

#test -z "$mode" && help
###denis test -n "$osxmin" && test -z "$vsdk" && help

TMOS=$(OsType)
# SDK path setting 
case "$TMOS" in
(mac|lin) sdk=$(dirname $PWD)/SDK;;
(win) sdk=/SDK; QT=/Qt;;
esac

mkdir -p $sdk/{bin,lib}

if test ! -x $sdk/bin/pkg-config -a -x $PWD/bin/pkg-config
then ln -s $PWD/bin/pkg-config $sdk/bin/pkg-config
fi

declare fdone=$PWD/${fgdb}mac${vsdk}_done.txt
test -f $fdone && source $fdone
setEnv

export -f absPath tmsdk status setStatus isDone CompilVersion SubstInFile checkflag
eval export $CMD
export LIBRARY_PATH=${LIBRARY_PATH:+ }$TMSDKPH/lib

case $mode in
m)
  echo switching to environment $sdk. Type exit to leave it
  echo Type tmsdk to remember its location
  cat <<EOF
To build the sdk, type in sequence :
./mksrc.sh     : for sources extration
./mklib.sh     : copy the system static libs for linking with
./mkbasic.sh   : to build the basic material
./mkgs.sh      : ghostscript building
./mkguile.sh   : guile building
./mkqt.sh      : for Qt building
EOF

  bash --norc
    ;;
a)
  for action in mktools.sh mkauto.sh mksrc.sh mkbasic.sh mkgs.sh mkguile.sh mkqt.sh mklib.sh     
  do 
    if test -n "$action"
    then 
      ./$action
    fi
  done
  status
  ;;
c) CheckPrereq; ret=$?
	test $ret -eq 0 && exit 100
  ./mktools.sh;./mkauto.sh
	test $ret -eq 2 && echo ==>type  Exit and reopen the mingw shell
	;;
S) status;;
*) help;;
esac

